<?php
// Set secure access flag for includes
define('SECURE_ACCESS', true);

// Include necessary security and configuration files
$basePath = dirname(__FILE__);
require_once($basePath . '/includes/config.php');

// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Set up logging function
function logDebug($message) {
    $logFile = __DIR__ . "/form-debug.log";
    
    // Create log with timestamp
    $timestamp = date('Y-m-d H:i:s');
    $logMessage = "[$timestamp] $message" . PHP_EOL;
    
    // Write to log file
    file_put_contents($logFile, $logMessage, FILE_APPEND);
}

// Start debug log
logDebug("Starting form debug script");

// Log relevant form information
echo "<h2>Career Form Debug Information</h2>";
echo "<p>This page helps diagnose issues with the career form submission process.</p>";

// Check if form processing scripts exist
echo "<h3>Form Processing Scripts</h3>";
echo "<ul>";
$attachmentScript = file_exists($basePath . '/mail/mail-send-attachment.php') ? "✅ Exists" : "❌ Missing";
$careerScript = file_exists($basePath . '/mail/mail-send-career.php') ? "✅ Exists" : "❌ Missing";
echo "<li>mail-send-attachment.php: $attachmentScript</li>";
echo "<li>mail-send-career.php: $careerScript</li>";
echo "</ul>";

// Check database connection
echo "<h3>Database Connection</h3>";
try {
    // Include database configuration
    require_once($basePath . '/mail/db_config.php');
    
    // Try to connect
    $conn = getDbConnection();
    
    if ($conn) {
        echo "<p>✅ Database connection successful</p>";
        
        // Check if per_resume table exists
        $result = $conn->query("SHOW TABLES LIKE 'per_resume'");
        if ($result->num_rows > 0) {
            echo "<p>✅ 'per_resume' table exists</p>";
            
            // Check table structure
            $result = $conn->query("DESCRIBE per_resume");
            echo "<p>Table Structure:</p><pre>";
            if ($result) {
                echo "<table border='1' cellpadding='5'>";
                echo "<tr><th>Field</th><th>Type</th><th>Null</th><th>Key</th><th>Default</th><th>Extra</th></tr>";
                while ($row = $result->fetch_assoc()) {
                    echo "<tr>";
                    echo "<td>" . $row['Field'] . "</td>";
                    echo "<td>" . $row['Type'] . "</td>";
                    echo "<td>" . $row['Null'] . "</td>";
                    echo "<td>" . $row['Key'] . "</td>";
                    echo "<td>" . $row['Default'] . "</td>";
                    echo "<td>" . $row['Extra'] . "</td>";
                    echo "</tr>";
                }
                echo "</table>";
            } else {
                echo "Unable to get table structure: " . $conn->error;
            }
            echo "</pre>";
            
            // Check latest entries
            $result = $conn->query("SELECT * FROM per_resume ORDER BY id DESC LIMIT 5");
            if ($result && $result->num_rows > 0) {
                echo "<p>Latest 5 entries:</p>";
                echo "<table border='1' cellpadding='5'>";
                echo "<tr><th>ID</th><th>Name</th><th>Email</th><th>Position</th><th>Upload Time</th></tr>";
                while ($row = $result->fetch_assoc()) {
                    echo "<tr>";
                    echo "<td>" . ($row['id'] ?? 'N/A') . "</td>";
                    echo "<td>" . ($row['Name'] ?? 'N/A') . "</td>";
                    echo "<td>" . ($row['Email'] ?? 'N/A') . "</td>";
                    echo "<td>" . ($row['Position_Name'] ?? 'N/A') . "</td>";
                    echo "<td>" . ($row['Upload_On'] ?? 'N/A') . "</td>";
                    echo "</tr>";
                }
                echo "</table>";
            } else {
                echo "<p>No entries found or error: " . $conn->error . "</p>";
            }
        } else {
            echo "<p>❌ 'per_resume' table not found</p>";
        }
        
        // Close connection
        $conn->close();
    } else {
        echo "<p>❌ Database connection failed</p>";
    }
} catch (Exception $e) {
    echo "<p>❌ Error: " . $e->getMessage() . "</p>";
}

// Form URL check
echo "<h3>Form Submission Target</h3>";
echo "<p>The form in solution_architect.php is targeting: <code>../mail/mail-send-attachment.php</code></p>";

// Check the hidden fields in the form
echo "<h3>Form Input Fields Needed:</h3>";
echo "<ul>";
echo "<li>csrf_token (hidden): Required for security validation</li>";
echo "<li>c_name (text): Form field 'name'</li>";
echo "<li>c_email (email): Form field 'email'</li>";
echo "<li>c_code (select): Form field 'code'</li>";
echo "<li>c_phone (text): Form field 'phone'</li>";
echo "<li>c_position (hidden): Set to position name</li>";
echo "<li>c_selected (hidden): Country selection</li>";
echo "<li>attachment[] (file): The resume upload</li>";
echo "</ul>";

// Compare the field names between the form and the PHP script
echo "<h3>Form vs Script Field Comparison</h3>";
echo "<table border='1' cellpadding='5'>";
echo "<tr><th>Career Form Field</th><th>mail-send-attachment.php Expects</th><th>mail-send-career.php Expects</th><th>Match?</th></tr>";
echo "<tr><td>name</td><td>name</td><td>c_name</td><td>❌ Mismatch</td></tr>";
echo "<tr><td>email</td><td>email</td><td>c_email</td><td>❌ Mismatch</td></tr>";
echo "<tr><td>code</td><td>code</td><td>c_code</td><td>❌ Mismatch</td></tr>";
echo "<tr><td>phone</td><td>phone</td><td>c_phone</td><td>❌ Mismatch</td></tr>";
echo "<tr><td>attachment[]</td><td>fileUploadimage</td><td>attachment[]</td><td>❌ Mismatch</td></tr>";
echo "<tr><td>c_position</td><td>job_id</td><td>c_position</td><td>❌ Mismatch</td></tr>";
echo "</table>";

echo "<h3>Conclusion</h3>";
echo "<p>The form is submitting to mail-send-attachment.php but the field names in the form match what mail-send-career.php expects. This is causing the data to not be properly processed.</p>";

echo "<h3>Recommended Fix</h3>";
echo "<p>Either:</p>";
echo "<ol>";
echo "<li>Update the form to use field names that match mail-send-attachment.php (name, email, etc.)</li>";
echo "<li>Update the JavaScript to point to mail-send-career.php instead of mail-send-attachment.php</li>";
echo "</ol>";

logDebug("Debug script completed");
?> 