<?php
/**
 * SRI Disabler
 * 
 * This script modifies sri-implementation.php to temporarily disable SRI checks 
 * by using fallback functions instead. This can help diagnose if the issues on
 * the website are related to SRI integrity validation.
 */

// Set headers and error reporting
header('Content-Type: text/html; charset=utf-8');
ini_set('display_errors', 1);
error_reporting(E_ALL);

// Define the path to sri-implementation.php
$sriFile = __DIR__ . '/security/sri-implementation.php';
if (!file_exists($sriFile)) {
    $sriFile = __DIR__ . '/sri-implementation.php';
}

// Check if the file exists
$fileExists = file_exists($sriFile);
$backupFile = $sriFile . '.backup';

// Define possible actions
$action = isset($_GET['action']) ? $_GET['action'] : '';
$message = '';
$success = false;

// Process actions
if ($action === 'disable' && $fileExists) {
    // Create a backup if it doesn't exist
    if (!file_exists($backupFile)) {
        copy($sriFile, $backupFile);
    }
    
    // Read the file
    $content = file_get_contents($sriFile);
    
    // Modify sriScriptTag function to use fallback
    $content = preg_replace(
        '/function sriScriptTag\([^)]*\)[^{]*{.*?return \'<script.*?\';/s',
        'function sriScriptTag($url, $defer = true, $async = false) {
        // Temporarily disabled SRI checks - using fallback
        return fallbackScriptTag($url, $defer, $async);',
        $content
    );
    
    // Modify sriStylesheetTag function to use fallback
    $content = preg_replace(
        '/function sriStylesheetTag\([^)]*\)[^{]*{.*?return \'<link.*?\';/s',
        'function sriStylesheetTag($url) {
        // Temporarily disabled SRI checks - using fallback
        return fallbackStylesheetTag($url);',
        $content
    );
    
    // Save the modified file
    $success = file_put_contents($sriFile, $content) !== false;
    $message = $success ? 'SRI checks have been successfully disabled.' : 'Failed to disable SRI checks.';
    
} else if ($action === 'restore' && file_exists($backupFile)) {
    // Restore from backup
    $success = copy($backupFile, $sriFile);
    $message = $success ? 'SRI checks have been successfully restored from backup.' : 'Failed to restore SRI checks.';
    
} else if ($action === 'status' && $fileExists) {
    // Check current status
    $content = file_get_contents($sriFile);
    if (strpos($content, 'Temporarily disabled SRI checks') !== false) {
        $message = 'SRI checks are currently DISABLED.';
        $status = 'disabled';
    } else {
        $message = 'SRI checks are currently ENABLED.';
        $status = 'enabled';
    }
    $success = true;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>SRI Disabler Tool</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            line-height: 1.6;
            padding: 20px;
            max-width: 800px;
            margin: 0 auto;
        }
        h1, h2 {
            color: #333;
        }
        .message {
            padding: 10px;
            border-radius: 5px;
            margin: 20px 0;
        }
        .success {
            background-color: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        .error {
            background-color: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        .info {
            background-color: #d1ecf1;
            color: #0c5460;
            border: 1px solid #bee5eb;
        }
        .warning {
            background-color: #fff3cd;
            color: #856404;
            border: 1px solid #ffeeba;
        }
        .actions {
            margin: 20px 0;
            display: flex;
            gap: 10px;
        }
        .btn {
            display: inline-block;
            padding: 8px 16px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            text-decoration: none;
            text-align: center;
        }
        .btn-primary {
            background-color: #007bff;
            color: white;
        }
        .btn-success {
            background-color: #28a745;
            color: white;
        }
        .btn-danger {
            background-color: #dc3545;
            color: white;
        }
        .btn-info {
            background-color: #17a2b8;
            color: white;
        }
        pre {
            background-color: #f5f5f5;
            padding: 15px;
            border-radius: 5px;
            overflow-x: auto;
        }
    </style>
</head>
<body>
    <h1>SRI Disabler Tool</h1>
    
    <div class="info message">
        <p>This tool allows you to temporarily disable SRI (Subresource Integrity) checks to diagnose if resource loading issues are related to integrity validation.</p>
    </div>
    
    <?php if (!$fileExists): ?>
        <div class="error message">
            <p>Error: Could not find sri-implementation.php file. Make sure it exists at one of these locations:</p>
            <ul>
                <li><?= htmlspecialchars(__DIR__ . '/security/sri-implementation.php') ?></li>
                <li><?= htmlspecialchars(__DIR__ . '/sri-implementation.php') ?></li>
            </ul>
        </div>
    <?php else: ?>
        <div class="success message">
            <p>Found SRI implementation file at: <?= htmlspecialchars($sriFile) ?></p>
        </div>
        
        <?php if ($message): ?>
            <div class="<?= $success ? 'success' : 'error' ?> message">
                <p><?= htmlspecialchars($message) ?></p>
            </div>
        <?php endif; ?>
        
        <div class="actions">
            <a href="?action=status" class="btn btn-info">Check Status</a>
            <a href="?action=disable" class="btn btn-danger">Disable SRI Checks</a>
            <?php if (file_exists($backupFile)): ?>
                <a href="?action=restore" class="btn btn-success">Restore SRI Checks</a>
            <?php endif; ?>
        </div>
        
        <h2>How This Works</h2>
        <p>When you disable SRI checks:</p>
        <ol>
            <li>A backup of your original sri-implementation.php file is created</li>
            <li>The <code>sriScriptTag</code> and <code>sriStylesheetTag</code> functions are modified to use the fallback functions without integrity checks</li>
            <li>This allows all resources to load without integrity validation</li>
        </ol>
        
        <div class="warning message">
            <p><strong>Important:</strong> Disabling SRI checks reduces security. This should only be done temporarily for diagnostic purposes.</p>
            <p>After testing, make sure to restore the original configuration by clicking "Restore SRI Checks".</p>
        </div>
        
        <h2>Next Steps After Disabling SRI</h2>
        <ol>
            <li>Refresh your website and check if the resources now load correctly</li>
            <li>If the website works with SRI disabled, the issue is likely with the integrity hashes</li>
            <li>Use the <code>fix-integrity.php</code> tool to generate updated hashes</li>
            <li>After fixing the hashes, restore the SRI checks using the "Restore SRI Checks" button</li>
        </ol>
    <?php endif; ?>
</body>
</html> 