<?php
// Start output buffering to prevent any unwanted output
ob_start();

// Helper function to check if a header exists and retrieve its value
function getHeaderValue($headers, $headerName) {
    $headerName = strtolower($headerName);
    foreach ($headers as $header => $value) {
        if (strtolower($header) === $headerName) {
            return $value;
        }
    }
    return null;
}

// Function to prettify the CSP string
function formatCSP($csp) {
    // Replace semicolons with semicolons followed by line breaks
    $formatted = str_replace('; ', ";\n", $csp);
    // Add indentation to each line except the first
    $formatted = str_replace(";\n", ";\n    ", $formatted);
    return $formatted;
}

// Set a custom header to verify the diagnostic is running
header('X-Diagnostic: CSP-Check');

// Capture the current headers
$allHeaders = [];
foreach (headers_list() as $header) {
    list($name, $value) = explode(':', $header, 2);
    $allHeaders[$name] = trim($value);
}

// Get the CSP header (could be either Content-Security-Policy or X-Content-Security-Policy)
$cspValue = getHeaderValue($allHeaders, 'Content-Security-Policy');
if (!$cspValue) {
    $cspValue = getHeaderValue($allHeaders, 'X-Content-Security-Policy');
}

// Clear the output buffer
ob_clean();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>CSP Diagnostic</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            line-height: 1.6;
            padding: 20px;
            max-width: 1200px;
            margin: 0 auto;
        }
        h1, h2 {
            color: #333;
        }
        pre {
            background-color: #f5f5f5;
            padding: 15px;
            border-radius: 5px;
            overflow-x: auto;
            white-space: pre-wrap;
            word-wrap: break-word;
        }
        .section {
            margin-bottom: 30px;
            padding: 20px;
            border: 1px solid #ddd;
            border-radius: 5px;
        }
        .success { color: green; }
        .warning { color: orange; }
        .error { color: red; }
        table {
            width: 100%;
            border-collapse: collapse;
            margin-bottom: 20px;
        }
        th, td {
            border: 1px solid #ddd;
            padding: 8px;
            text-align: left;
        }
        th {
            background-color: #f2f2f2;
        }
        tr:nth-child(even) {
            background-color: #f9f9f9;
        }
    </style>
</head>
<body>
    <h1>Content Security Policy Diagnostic</h1>
    
    <div class="section">
        <h2>Current CSP Header</h2>
        <?php if ($cspValue): ?>
            <pre><?= htmlspecialchars(formatCSP($cspValue)) ?></pre>
        <?php else: ?>
            <p class="error">No Content Security Policy header found!</p>
        <?php endif; ?>
    </div>
    
    <div class="section">
        <h2>All HTTP Headers</h2>
        <table>
            <tr>
                <th>Header</th>
                <th>Value</th>
            </tr>
            <?php foreach ($allHeaders as $name => $value): ?>
            <tr>
                <td><?= htmlspecialchars($name) ?></td>
                <td>
                    <?php if (strlen($value) > 100): ?>
                        <div style="max-height: 100px; overflow-y: auto;">
                            <?= htmlspecialchars($value) ?>
                        </div>
                    <?php else: ?>
                        <?= htmlspecialchars($value) ?>
                    <?php endif; ?>
                </td>
            </tr>
            <?php endforeach; ?>
        </table>
    </div>
    
    <div class="section">
        <h2>CSP Violation Detection</h2>
        <div id="violations">
            <p>No CSP violations detected yet. If any occur, they will appear here.</p>
        </div>
        <script>
            document.addEventListener('securitypolicyviolation', function(e) {
                var violationsDiv = document.getElementById('violations');
                violationsDiv.innerHTML = '';
                
                var violationInfo = document.createElement('div');
                violationInfo.innerHTML = 
                    '<h3 class="error">CSP Violation Detected</h3>' +
                    '<p><strong>Blocked URI:</strong> ' + e.blockedURI + '</p>' +
                    '<p><strong>Violated Directive:</strong> ' + e.violatedDirective + '</p>' +
                    '<p><strong>Original Policy:</strong> ' + e.originalPolicy + '</p>';
                
                violationsDiv.appendChild(violationInfo);
            });
        </script>
    </div>
    
    <div class="section">
        <h2>Resource Loading Test</h2>
        <p>Testing loading of common external resources:</p>
        
        <h3>1. External Stylesheets</h3>
        <ul id="stylesheet-test">Loading...</ul>
        
        <h3>2. External Scripts</h3>
        <ul id="script-test">Loading...</ul>
        
        <h3>3. Font Resources</h3>
        <ul id="font-test">Loading...</ul>
        
        <script>
            // Test various resources
            function testResources() {
                // Test stylesheets
                var stylesheets = [
                    { url: 'https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css', name: 'Bootstrap CSS' },
                    { url: 'https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css', name: 'Font Awesome' },
                    { url: 'https://cdnjs.cloudflare.com/ajax/libs/bootstrap-datepicker/1.9.0/css/bootstrap-datepicker.min.css', name: 'Bootstrap Datepicker CSS' }
                ];
                
                var scriptsToTest = [
                    { url: 'https://code.jquery.com/jquery-3.6.0.min.js', name: 'jQuery' },
                    { url: 'https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.min.js', name: 'Bootstrap JS' },
                    { url: 'https://cdnjs.cloudflare.com/ajax/libs/bootstrap-datepicker/1.9.0/js/bootstrap-datepicker.min.js', name: 'Bootstrap Datepicker JS' },
                    { url: 'https://www.googletagmanager.com/gtag/js?id=UA-XXXXX-Y', name: 'Google Analytics' }
                ];
                
                // Test stylesheets
                var stylesheetList = document.getElementById('stylesheet-test');
                stylesheetList.innerHTML = '';
                
                stylesheets.forEach(function(sheet) {
                    var link = document.createElement('link');
                    link.rel = 'stylesheet';
                    link.href = sheet.url;
                    
                    var listItem = document.createElement('li');
                    
                    link.onload = function() {
                        listItem.innerHTML = sheet.name + ': <span class="success">Loaded successfully</span>';
                    };
                    
                    link.onerror = function() {
                        listItem.innerHTML = sheet.name + ': <span class="error">Failed to load</span>';
                    };
                    
                    document.head.appendChild(link);
                    listItem.innerHTML = sheet.name + ': <span class="warning">Testing...</span>';
                    stylesheetList.appendChild(listItem);
                });
                
                // Test scripts
                var scriptList = document.getElementById('script-test');
                scriptList.innerHTML = '';
                
                scriptsToTest.forEach(function(scriptInfo) {
                    var listItem = document.createElement('li');
                    listItem.innerHTML = scriptInfo.name + ': <span class="warning">Testing...</span>';
                    scriptList.appendChild(listItem);
                    
                    var script = document.createElement('script');
                    script.src = scriptInfo.url;
                    
                    script.onload = function() {
                        listItem.innerHTML = scriptInfo.name + ': <span class="success">Loaded successfully</span>';
                    };
                    
                    script.onerror = function() {
                        listItem.innerHTML = scriptInfo.name + ': <span class="error">Failed to load</span>';
                    };
                    
                    document.body.appendChild(script);
                });
                
                // Test fonts
                var fontList = document.getElementById('font-test');
                fontList.innerHTML = '';
                
                // Google Fonts
                var fontLink = document.createElement('link');
                fontLink.rel = 'stylesheet';
                fontLink.href = 'https://fonts.googleapis.com/css2?family=Roboto:wght@400;700&display=swap';
                
                var fontListItem = document.createElement('li');
                fontListItem.innerHTML = 'Google Fonts (Roboto): <span class="warning">Testing...</span>';
                fontList.appendChild(fontListItem);
                
                fontLink.onload = function() {
                    // Create a test element with the font
                    var testElement = document.createElement('span');
                    testElement.style.fontFamily = "'Roboto', sans-serif";
                    testElement.style.visibility = 'hidden';
                    testElement.textContent = 'Test';
                    document.body.appendChild(testElement);
                    
                    // Check if the font is applied
                    setTimeout(function() {
                        fontListItem.innerHTML = 'Google Fonts (Roboto): <span class="success">Loaded successfully</span>';
                        document.body.removeChild(testElement);
                    }, 100);
                };
                
                fontLink.onerror = function() {
                    fontListItem.innerHTML = 'Google Fonts (Roboto): <span class="error">Failed to load</span>';
                };
                
                document.head.appendChild(fontLink);
            }
            
            // Run the tests after page load
            window.onload = testResources;
        </script>
    </div>
</body>
</html> 