<?php
/**
 * SRI Integrity Issue Fixer
 * 
 * This script helps identify and fix SRI integrity issues by:
 * 1. Testing common resources that might be failing due to integrity checks
 * 2. Regenerating hashes for those resources
 * 3. Providing code to update the sri-implementation.php file
 */

// Start session if not already started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Set the content type and error reporting
header('Content-Type: text/html; charset=utf-8');
ini_set('display_errors', 1);
error_reporting(E_ALL);

// Define common resources that might need their SRI hashes updated
$resources = [
    // Bootstrap Datepicker
    [
        'name' => 'Bootstrap Datepicker JS',
        'url' => 'https://cdnjs.cloudflare.com/ajax/libs/bootstrap-datepicker/1.9.0/js/bootstrap-datepicker.min.js',
        'type' => 'script'
    ],
    [
        'name' => 'Bootstrap Datepicker CSS',
        'url' => 'https://cdnjs.cloudflare.com/ajax/libs/bootstrap-datepicker/1.9.0/css/bootstrap-datepicker.min.css',
        'type' => 'stylesheet'
    ],
    // Font Awesome
    [
        'name' => 'Font Awesome CSS',
        'url' => 'https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css',
        'type' => 'stylesheet'
    ],
    // jQuery
    [
        'name' => 'jQuery',
        'url' => 'https://code.jquery.com/jquery-3.6.0.min.js',
        'type' => 'script'
    ],
    // Bootstrap
    [
        'name' => 'Bootstrap CSS',
        'url' => 'https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css',
        'type' => 'stylesheet'
    ],
    [
        'name' => 'Bootstrap JS',
        'url' => 'https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.min.js',
        'type' => 'script'
    ]
];

/**
 * Generate an SRI hash for a given URL
 * @param string $url The URL of the resource to hash
 * @return string|bool The SRI hash or false on failure
 */
function generateHash($url) {
    $content = @file_get_contents($url);
    if ($content === false) {
        return false;
    }
    
    $hash = base64_encode(hash('sha384', $content, true));
    return "sha384-$hash";
}

// Process form submission
$results = [];
$updatedHashes = [];
$errorMessages = [];

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['regenerate'])) {
    foreach ($resources as $resource) {
        $url = $resource['url'];
        $hash = generateHash($url);
        
        if ($hash !== false) {
            $results[$url] = [
                'name' => $resource['name'],
                'url' => $url,
                'hash' => $hash,
                'status' => 'success'
            ];
            $updatedHashes[$url] = $hash;
        } else {
            $results[$url] = [
                'name' => $resource['name'],
                'url' => $url,
                'hash' => 'Failed to generate',
                'status' => 'error'
            ];
            $errorMessages[] = "Failed to generate hash for {$resource['name']} ({$url})";
        }
    }
}

// Function to get updated code for sri-implementation.php
function getUpdatedHashesCode($hashes) {
    $code = "static \$hashCache = [\n";
    $code .= "    // jQuery\n";
    $code .= "    'https://code.jquery.com/jquery-3.6.0.min.js' => '{$hashes['https://code.jquery.com/jquery-3.6.0.min.js']}',\n";
    $code .= "    'https://code.jquery.com/jquery-3.5.1.min.js' => 'sha384-ZvpUoO/+PpLXR1lu4jmpXWu80pZlYUAfxl5NsBMWOEPSjUn/6Z/hRTt8+pR6L4N2',\n";
    $code .= "    \n";
    $code .= "    // Bootstrap\n";
    $code .= "    'https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.min.js' => '{$hashes['https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.min.js']}',\n";
    $code .= "    'https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css' => '{$hashes['https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css']}',\n";
    $code .= "    'https://cdn.jsdelivr.net/npm/bootstrap@5.0.2/dist/js/bootstrap.min.js' => 'sha384-cVKIPhGWiC2Al4u+LWgxfKTRIcfu0JTxR+EQDz/bgldoEyl4H0zUF0QKbrJ0EcQF',\n";
    $code .= "    'https://cdn.jsdelivr.net/npm/bootstrap@5.0.2/dist/css/bootstrap.min.css' => 'sha384-EVSTQN3/azprG1Anm3QDgpJLIm9Nao0Yz1ztcQTwFspd3yD65VohhpuuCOmLASjC',\n";
    $code .= "    \n";
    $code .= "    // Font Awesome\n";
    $code .= "    'https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css' => '{$hashes['https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css']}',\n";
    $code .= "    \n";
    $code .= "    // Bootstrap Datepicker\n";
    $code .= "    'https://cdnjs.cloudflare.com/ajax/libs/bootstrap-datepicker/1.9.0/js/bootstrap-datepicker.min.js' => '{$hashes['https://cdnjs.cloudflare.com/ajax/libs/bootstrap-datepicker/1.9.0/js/bootstrap-datepicker.min.js']}',\n";
    $code .= "    'https://cdnjs.cloudflare.com/ajax/libs/bootstrap-datepicker/1.9.0/css/bootstrap-datepicker.min.css' => '{$hashes['https://cdnjs.cloudflare.com/ajax/libs/bootstrap-datepicker/1.9.0/css/bootstrap-datepicker.min.css']}',\n";
    $code .= "];";
    
    return $code;
}

// Manual resource testing
$testResults = [];
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['test'])) {
    $testUrl = $_POST['test_url'];
    $testType = $_POST['test_type'];
    
    // Generate hash
    $hash = generateHash($testUrl);
    
    if ($hash !== false) {
        $testResults = [
            'url' => $testUrl,
            'hash' => $hash,
            'status' => 'success'
        ];
    } else {
        $testResults = [
            'url' => $testUrl,
            'status' => 'error',
            'message' => 'Failed to fetch resource or generate hash'
        ];
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>SRI Integrity Issue Fixer</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            line-height: 1.6;
            padding: 20px;
            max-width: 1200px;
            margin: 0 auto;
        }
        h1, h2, h3 {
            color: #333;
        }
        pre {
            background-color: #f5f5f5;
            padding: 15px;
            border-radius: 5px;
            overflow-x: auto;
        }
        .section {
            margin-bottom: 30px;
            padding: 20px;
            border: 1px solid #ddd;
            border-radius: 5px;
        }
        .success {
            color: green;
        }
        .error {
            color: red;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin-bottom: 20px;
        }
        th, td {
            border: 1px solid #ddd;
            padding: 8px;
            text-align: left;
        }
        th {
            background-color: #f2f2f2;
        }
        form {
            margin-bottom: 20px;
        }
        button {
            padding: 8px 16px;
            background-color: #4CAF50;
            color: white;
            border: none;
            border-radius: 4px;
            cursor: pointer;
        }
        button:hover {
            background-color: #45a049;
        }
        input[type="text"], select {
            padding: 8px;
            width: 100%;
            box-sizing: border-box;
            margin-bottom: 10px;
        }
    </style>
</head>
<body>
    <h1>SRI Integrity Issue Fixer</h1>
    
    <div class="section">
        <h2>About This Tool</h2>
        <p>This tool helps identify and fix SRI (Subresource Integrity) issues by:</p>
        <ol>
            <li>Regenerating SRI hashes for common resources</li>
            <li>Providing updated code for your sri-implementation.php file</li>
            <li>Testing specific resources that may be causing issues</li>
        </ol>
    </div>
    
    <div class="section">
        <h2>Regenerate Common SRI Hashes</h2>
        <p>Click the button below to regenerate SRI hashes for common resources:</p>
        
        <form method="post" action="">
            <button type="submit" name="regenerate" value="1">Regenerate All Hashes</button>
        </form>
        
        <?php if (!empty($results)): ?>
            <h3>Results:</h3>
            <?php if (!empty($errorMessages)): ?>
                <div class="error">
                    <strong>Errors:</strong>
                    <ul>
                        <?php foreach ($errorMessages as $error): ?>
                            <li><?= htmlspecialchars($error) ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            <?php endif; ?>
            
            <table>
                <tr>
                    <th>Resource</th>
                    <th>URL</th>
                    <th>Generated Hash</th>
                    <th>Status</th>
                </tr>
                <?php foreach ($results as $result): ?>
                    <tr>
                        <td><?= htmlspecialchars($result['name']) ?></td>
                        <td><?= htmlspecialchars($result['url']) ?></td>
                        <td><?= htmlspecialchars($result['hash']) ?></td>
                        <td class="<?= $result['status'] ?>">
                            <?= $result['status'] === 'success' ? 'Success' : 'Failed' ?>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </table>
            
            <h3>Updated Code for sri-implementation.php:</h3>
            <p>Replace the $hashCache array in your sri-implementation.php file with this code:</p>
            <pre><?= htmlspecialchars(getUpdatedHashesCode($updatedHashes)) ?></pre>
        <?php endif; ?>
    </div>
    
    <div class="section">
        <h2>Test Custom Resource</h2>
        <p>Test a specific resource to generate its SRI hash:</p>
        
        <form method="post" action="">
            <div>
                <label for="test_url">Resource URL:</label>
                <input type="text" id="test_url" name="test_url" required placeholder="https://...">
            </div>
            <div>
                <label for="test_type">Resource Type:</label>
                <select id="test_type" name="test_type">
                    <option value="script">JavaScript</option>
                    <option value="stylesheet">CSS</option>
                </select>
            </div>
            <button type="submit" name="test" value="1">Test Resource</button>
        </form>
        
        <?php if (!empty($testResults)): ?>
            <h3>Test Results:</h3>
            <?php if ($testResults['status'] === 'success'): ?>
                <div class="success">
                    <p><strong>Resource:</strong> <?= htmlspecialchars($testResults['url']) ?></p>
                    <p><strong>Generated Hash:</strong> <?= htmlspecialchars($testResults['hash']) ?></p>
                </div>
                <p>Add this to your sri-implementation.php file:</p>
                <pre>'<?= htmlspecialchars($testResults['url']) ?>' => '<?= htmlspecialchars($testResults['hash']) ?>',</pre>
            <?php else: ?>
                <div class="error">
                    <p><strong>Error:</strong> <?= htmlspecialchars($testResults['message']) ?></p>
                    <p><strong>Resource:</strong> <?= htmlspecialchars($testResults['url']) ?></p>
                </div>
            <?php endif; ?>
        <?php endif; ?>
    </div>
    
    <div class="section">
        <h2>Instructions for Fixing SRI Issues</h2>
        <ol>
            <li>First, regenerate all hashes using the "Regenerate All Hashes" button above.</li>
            <li>Copy the updated code and replace the <code>$hashCache</code> array in your <code>sri-implementation.php</code> file.</li>
            <li>If you have custom resources that are failing, test them individually using the "Test Custom Resource" section.</li>
            <li>After updating the hashes, upload the modified <code>sri-implementation.php</code> file to your server.</li>
            <li>Clear your browser cache and test your website again.</li>
        </ol>
        
        <h3>Alternative Approach: Temporary Disable SRI</h3>
        <p>If you continue to face issues, you can temporarily disable SRI by editing your <code>sri-implementation.php</code> file:</p>
        <ol>
            <li>Modify the <code>sriScriptTag</code> and <code>sriStylesheetTag</code> functions to use <code>fallbackScriptTag</code> and <code>fallbackStylesheetTag</code> instead.</li>
            <li>This will allow your website to function while you investigate the integrity issues further.</li>
        </ol>
    </div>
</body>
</html> 