<?php
/**
 * Security-related functions
 * Contains functions for CSRF protection, input validation, etc.
 */

// Prevent direct access to this file
if (!defined('SECURE_ACCESS') && basename($_SERVER['PHP_SELF']) == basename(__FILE__)) {
    header('HTTP/1.0 403 Forbidden');
    exit('Direct access to this file is not allowed.');
}

/**
 * Generate a new CSRF token
 * 
 * @return string The generated token
 */
function generateCSRFToken() {
    // Start session if not started
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }
    
    // Generate a new token if one doesn't exist or is expired
    if (
        !isset($_SESSION['csrf_token']) || 
        !isset($_SESSION['csrf_token_time']) || 
        (time() - $_SESSION['csrf_token_time'] > CSRF_EXPIRATION)
    ) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
        $_SESSION['csrf_token_time'] = time();
    }
    
    return $_SESSION['csrf_token'];
}

/**
 * Validate a CSRF token
 * 
 * @param string $token The token to validate
 * @return bool Whether the token is valid
 */
function validateCSRFToken($token) {
    // Start session if not started
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }
    
    // Check if token exists in session
    if (!isset($_SESSION['csrf_token'])) {
        securityLog("CSRF token not found in session", "WARNING");
        return false;
    }
    
    // Check if token is empty
    if (empty($token)) {
        securityLog("Empty CSRF token submitted", "WARNING");
        return false;
    }
    
    // Check if token is valid
    if (!hash_equals($_SESSION['csrf_token'], $token)) {
        securityLog("Invalid CSRF token: " . $token, "WARNING");
        return false;
    }
    
    // Check if token is expired
    if (
        isset($_SESSION['csrf_token_time']) && 
        (time() - $_SESSION['csrf_token_time'] > CSRF_EXPIRATION)
    ) {
        securityLog("Expired CSRF token", "WARNING");
        return false;
    }
    
    return true;
}

/**
 * Log security events
 * 
 * @param string $message The message to log
 * @param string $level The level (INFO, WARNING, ERROR)
 * @return void
 */
function securityLog($message, $level = 'INFO') {
    $timestamp = date('Y-m-d H:i:s');
    $ipAddress = isset($_SERVER['REMOTE_ADDR']) ? $_SERVER['REMOTE_ADDR'] : 'Unknown';
    $userAgent = isset($_SERVER['HTTP_USER_AGENT']) ? $_SERVER['HTTP_USER_AGENT'] : 'Unknown';
    
    $logEntry = "[{$timestamp}] [{$level}] [{$ipAddress}] [{$userAgent}] {$message}" . PHP_EOL;
    
    // Log file path
    $logFile = dirname(__DIR__) . '/logs/security.log';
    
    // Create directory if it doesn't exist
    if (!is_dir(dirname($logFile))) {
        mkdir(dirname($logFile), 0755, true);
    }
    
    file_put_contents($logFile, $logEntry, FILE_APPEND);
}

/**
 * Sanitize input
 * 
 * @param string $input The input to sanitize
 * @return string Sanitized input
 */
function sanitizeInput($input) {
    $input = trim($input);
    $input = stripslashes($input);
    $input = htmlspecialchars($input, ENT_QUOTES, 'UTF-8');
    return $input;
}

/**
 * Validate email address
 * 
 * @param string $email The email to validate
 * @return bool Whether the email is valid
 */
function validateEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

/**
 * Validate phone number
 * 
 * @param string $phone The phone number to validate
 * @return bool Whether the phone number is valid
 */
function validatePhone($phone) {
    // Basic pattern for international phone numbers
    return preg_match('/^[0-9\+\-\(\) ]{6,20}$/', $phone) === 1;
}

/**
 * Validate file upload
 * 
 * @param array $file The $_FILES array element
 * @param array $allowedTypes Allowed MIME types
 * @param int $maxSize Maximum file size in bytes
 * @return array|bool Returns array with info if valid, false otherwise
 */
function validateFileUpload($file, $allowedTypes = [], $maxSize = 0) {
    // Check if file exists and there are no errors
    if (!isset($file) || $file['error'] !== UPLOAD_ERR_OK) {
        securityLog("File upload error: " . ($file['error'] ?? 'Unknown'), "WARNING");
        return false;
    }
    
    // Use default allowed types and max size if not provided
    if (empty($allowedTypes) && defined('ALLOWED_EXTENSIONS')) {
        // Convert extensions to MIME types
        $extensionToMime = [
            'pdf' => 'application/pdf',
            'doc' => 'application/msword',
            'docx' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document'
        ];
        
        $allowedTypes = [];
        foreach (ALLOWED_EXTENSIONS as $ext) {
            if (isset($extensionToMime[$ext])) {
                $allowedTypes[] = $extensionToMime[$ext];
            }
        }
    }
    
    if ($maxSize === 0 && defined('MAX_UPLOAD_SIZE')) {
        $maxSize = MAX_UPLOAD_SIZE;
    }
    
    // Check file size
    if ($file['size'] > $maxSize) {
        securityLog("File too large: " . $file['size'] . " bytes", "WARNING");
        return false;
    }
    
    // Check file type
    $finfo = new finfo(FILEINFO_MIME_TYPE);
    $mimeType = $finfo->file($file['tmp_name']);
    
    if (!empty($allowedTypes) && !in_array($mimeType, $allowedTypes)) {
        securityLog("Invalid file type: " . $mimeType, "WARNING");
        return false;
    }
    
    // Generate a secure filename
    $fileInfo = pathinfo($file['name']);
    $extension = strtolower($fileInfo['extension']);
    $newFilename = 'file_' . time() . '_' . bin2hex(random_bytes(8)) . '.' . $extension;
    
    return [
        'original_name' => $file['name'],
        'mime_type' => $mimeType,
        'size' => $file['size'],
        'tmp_name' => $file['tmp_name'],
        'new_filename' => $newFilename
    ];
}

// Define secure access constant to allow inclusion
define('SECURE_ACCESS', true);
?> 