<?php
/**
 * jQuery Helper Script
 * 
 * This script helps identify and resolve jQuery reference errors by:
 * 1. Checking for jQuery loading issues
 * 2. Ensuring jQuery is loaded before other scripts
 * 3. Providing a jQuery compatibility mode
 */

// Set content type
header('Content-Type: text/html; charset=utf-8');
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>jQuery Reference Error Fix</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            line-height: 1.6;
            max-width: 1000px;
            margin: 0 auto;
            padding: 20px;
        }
        h1, h2, h3 {
            color: #333;
        }
        .card {
            border: 1px solid #ddd;
            border-radius: 4px;
            padding: 20px;
            margin-bottom: 20px;
            background-color: #f9f9f9;
        }
        .success {
            color: green;
            font-weight: bold;
        }
        .error {
            color: red;
            font-weight: bold;
        }
        .code {
            background-color: #f0f0f0;
            border: 1px solid #ccc;
            border-radius: 4px;
            padding: 15px;
            font-family: monospace;
            white-space: pre-wrap;
            overflow-x: auto;
            margin: 10px 0;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin: 15px 0;
        }
        th, td {
            border: 1px solid #ddd;
            padding: 8px;
            text-align: left;
        }
        th {
            background-color: #f2f2f2;
        }
    </style>
</head>
<body>
    <h1>jQuery Reference Error Fix</h1>
    
    <div class="card">
        <h2>Common Issue: <code>Uncaught ReferenceError: $ is not defined</code></h2>
        <p>This error occurs when JavaScript code attempts to use jQuery's <code>$</code> symbol before jQuery is fully loaded.</p>
    </div>
    
    <div class="card">
        <h2>jQuery Detection Test</h2>
        <div id="jquery-detection-results">Testing jQuery availability...</div>
        
        <script>
            // This script checks if jQuery is already available
            document.addEventListener('DOMContentLoaded', function() {
                var results = document.getElementById('jquery-detection-results');
                
                if (typeof jQuery !== 'undefined') {
                    results.innerHTML = '<p class="success">jQuery is available (version: ' + jQuery.fn.jquery + ')</p>';
                } else {
                    results.innerHTML = '<p class="error">jQuery is NOT available on this page</p>';
                    
                    // Add jQuery for testing
                    var script = document.createElement('script');
                    script.src = 'https://code.jquery.com/jquery-3.6.0.min.js';
                    script.integrity = 'sha384-vtXRMe3mGCbOeY7l30aIg8H9p3GdeSe4IFlP6G8JMa7o7lXvnz3GFKzPxzJdPfGK';
                    script.crossOrigin = 'anonymous';
                    
                    script.onload = function() {
                        results.innerHTML += '<p class="success">jQuery was loaded by this tool (version: ' + jQuery.fn.jquery + ')</p>';
                        checkJQueryUsage();
                    };
                    
                    document.head.appendChild(script);
                }
            });
            
            function checkJQueryUsage() {
                // Check all script tags on the page
                var scripts = document.getElementsByTagName('script');
                var inlineScripts = [];
                
                for (var i = 0; i < scripts.length; i++) {
                    // Check for inline scripts
                    if (!scripts[i].src && scripts[i].innerHTML.indexOf('$') !== -1) {
                        inlineScripts.push({
                            index: i,
                            content: scripts[i].innerHTML.trim().substring(0, 100) + (scripts[i].innerHTML.length > 100 ? '...' : '')
                        });
                    }
                }
                
                if (inlineScripts.length > 0) {
                    var usageResults = document.createElement('div');
                    usageResults.innerHTML = '<h3>Found ' + inlineScripts.length + ' inline scripts using jQuery:</h3>';
                    
                    var table = document.createElement('table');
                    table.innerHTML = '<tr><th>Index</th><th>Content Preview</th></tr>';
                    
                    inlineScripts.forEach(function(script) {
                        var row = document.createElement('tr');
                        row.innerHTML = '<td>' + script.index + '</td><td><code>' + escapeHtml(script.content) + '</code></td>';
                        table.appendChild(row);
                    });
                    
                    usageResults.appendChild(table);
                    document.getElementById('jquery-detection-results').appendChild(usageResults);
                }
            }
            
            function escapeHtml(unsafe) {
                return unsafe
                    .replace(/&/g, "&amp;")
                    .replace(/</g, "&lt;")
                    .replace(/>/g, "&gt;")
                    .replace(/"/g, "&quot;")
                    .replace(/'/g, "&#039;");
            }
        </script>
    </div>
    
    <div class="card">
        <h2>Solutions</h2>
        
        <h3>1. Ensure jQuery is loaded before dependent scripts</h3>
        <p>Make sure jQuery is loaded before any scripts that use it. Move the jQuery script tag above other scripts or add the <code>defer</code> attribute to ensure proper loading order.</p>
        <div class="code">&lt;script src="https://code.jquery.com/jquery-3.6.0.min.js" 
        integrity="sha384-vtXRMe3mGCbOeY7l30aIg8H9p3GdeSe4IFlP6G8JMa7o7lXvnz3GFKzPxzJdPfGK" 
        crossorigin="anonymous"&gt;&lt;/script&gt;

&lt;!-- Other scripts that depend on jQuery should come AFTER jQuery --&gt;
&lt;script src="your-script.js"&gt;&lt;/script&gt;</div>
        
        <h3>2. Use document.ready wrapper</h3>
        <p>Wrap jQuery code in a document.ready function to ensure it runs after the DOM is fully loaded:</p>
        <div class="code">$(document).ready(function() {
    // Your jQuery code here
});</div>
        
        <h3>3. Use jQuery's noConflict mode</h3>
        <p>If multiple libraries are using the $ symbol, use jQuery's noConflict mode:</p>
        <div class="code">var jq = jQuery.noConflict();
// Now use jq instead of $ 
jq(document).ready(function() {
    jq("#element").hide();
});</div>
        
        <h3>4. Add a compatibility fix</h3>
        <p>Add this code snippet to the top of your scripts that use jQuery:</p>
        <div class="code">// jQuery compatibility fix
if (typeof jQuery === 'undefined') {
    console.warn('jQuery is not loaded yet. Adding compatibility layer.');
    // Create temporary $ function until jQuery loads
    window.$ = function() {
        var args = arguments;
        var ready = false;
        // Queue the call to run when jQuery is available
        var interval = setInterval(function() {
            if (typeof jQuery !== 'undefined') {
                clearInterval(interval);
                jQuery.apply(jQuery, args);
            }
        }, 50);
    };
}</div>
    </div>
    
    <div class="card">
        <h2>Script Loading Analysis</h2>
        <div id="script-analysis">
            <p>Analyzing scripts on this page...</p>
        </div>
        
        <script>
            document.addEventListener('DOMContentLoaded', function() {
                setTimeout(function() {
                    analyzeScripts();
                }, 1000); // Wait a bit to ensure all scripts are loaded
            });
            
            function analyzeScripts() {
                var scripts = document.getElementsByTagName('script');
                var externalScripts = [];
                
                for (var i = 0; i < scripts.length; i++) {
                    if (scripts[i].src) {
                        externalScripts.push({
                            index: i,
                            src: scripts[i].src,
                            async: scripts[i].async,
                            defer: scripts[i].defer
                        });
                    }
                }
                
                var analysisElement = document.getElementById('script-analysis');
                if (externalScripts.length > 0) {
                    analysisElement.innerHTML = '<h3>Found ' + externalScripts.length + ' external scripts:</h3>';
                    
                    var table = document.createElement('table');
                    table.innerHTML = '<tr><th>Index</th><th>Source</th><th>Async</th><th>Defer</th></tr>';
                    
                    externalScripts.forEach(function(script) {
                        var row = document.createElement('tr');
                        row.innerHTML = 
                            '<td>' + script.index + '</td>' +
                            '<td>' + script.src + '</td>' +
                            '<td>' + (script.async ? 'Yes' : 'No') + '</td>' +
                            '<td>' + (script.defer ? 'Yes' : 'No') + '</td>';
                        table.appendChild(row);
                    });
                    
                    analysisElement.appendChild(table);
                    
                    // Check for jQuery
                    var jQueryScripts = externalScripts.filter(function(script) {
                        return script.src.indexOf('jquery') !== -1;
                    });
                    
                    if (jQueryScripts.length > 0) {
                        var jqueryInfo = document.createElement('div');
                        jqueryInfo.innerHTML = '<h3>jQuery Loading:</h3>';
                        jqueryInfo.innerHTML += '<p>jQuery script found at index ' + jQueryScripts[0].index + '</p>';
                        
                        if (jQueryScripts[0].defer) {
                            jqueryInfo.innerHTML += '<p class="success">jQuery has the defer attribute, which helps ensure proper loading order.</p>';
                        } else {
                            jqueryInfo.innerHTML += '<p class="error">jQuery does not have the defer attribute. Consider adding it for better loading.</p>';
                        }
                        
                        analysisElement.appendChild(jqueryInfo);
                    }
                } else {
                    analysisElement.innerHTML = '<p>No external scripts found on this page.</p>';
                }
            }
        </script>
    </div>
    
    <div class="card">
        <h2>jQuery Version Compatibility</h2>
        <p>Different websites may use different jQuery versions. Here's a compatibility guide:</p>
        <table>
            <tr>
                <th>jQuery Version</th>
                <th>Browser Support</th>
                <th>Notes</th>
            </tr>
            <tr>
                <td>3.x</td>
                <td>Modern browsers</td>
                <td>No support for IE 6-8</td>
            </tr>
            <tr>
                <td>2.x</td>
                <td>Modern browsers</td>
                <td>No support for IE 6-8</td>
            </tr>
            <tr>
                <td>1.12.x / 1.11.x</td>
                <td>All browsers including IE 6-8</td>
                <td>Use for legacy applications</td>
            </tr>
        </table>
    </div>
</body>
</html> 