<?php
// Set secure access flag for includes
define('SECURE_ACCESS', true);

// Include necessary security and configuration files
$basePath = dirname(dirname(__FILE__));
require_once($basePath . '/includes/config.php');
require_once($basePath . '/includes/security.php');

// Enable error reporting for debugging (remove in production)
// error_reporting(E_ALL);
// ini_set('display_errors', 1);

// Log function for security related events
function securityLog($message, $level = 'INFO') {
    $logFile = __DIR__ . "/../logs/security.log";
    $logDir = dirname($logFile);
    
    // Create logs directory if it doesn't exist
    if (!file_exists($logDir)) {
        mkdir($logDir, 0755, true);
    }
    
    $timestamp = date('Y-m-d H:i:s');
    $logMessage = "[$timestamp] [$level] $message" . PHP_EOL;
    error_log($logMessage, 3, $logFile);
}

// Validate CSRF token
function validateCSRFToken($token) {
    if (!isset($token)) {
        securityLog("CSRF token missing in form submission", "WARNING");
        return false;
    }
    
    $tokenValid = false;
    
    // If session based tokens
    if (isset($_SESSION['csrf_token'])) {
        $tokenValid = ($token === $_SESSION['csrf_token']);
    } 
    // Fallback to other validation methods if needed
    else if (function_exists('validateToken')) {
        $tokenValid = validateToken($token);
    }
    
    if (!$tokenValid) {
        securityLog("Invalid CSRF token: " . $token, "WARNING");
    }
    
    return $tokenValid;
}

// Sanitize input function
function sanitizeInput($input) {
    $input = trim($input);
    $input = stripslashes($input);
    $input = htmlspecialchars($input, ENT_QUOTES, 'UTF-8');
    return $input;
}

// Initialize response
$response = 0;

// Start session if not already started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Check if request method is POST
if ($_SERVER["REQUEST_METHOD"] != "POST") {
    securityLog("Invalid request method: " . $_SERVER["REQUEST_METHOD"], "WARNING");
    die(json_encode(["error" => "Invalid request method"]));
}

// Validate CSRF token
if (!isset($_POST['csrf_token']) || !validateCSRFToken($_POST['csrf_token'])) {
    // CSRF token validation failed
    securityLog("CSRF token validation failed", "WARNING");
    echo '1'; // Error response
    exit;
}

// Validate and sanitize input data
$name = isset($_POST['name']) ? sanitizeInput($_POST['name']) : '';
$email = isset($_POST['email']) ? filter_var(sanitizeInput($_POST['email']), FILTER_SANITIZE_EMAIL) : '';
$code = isset($_POST['code']) ? sanitizeInput($_POST['code']) : '';
$phone = isset($_POST['phone']) ? sanitizeInput($_POST['phone']) : '';
$job_id = isset($_POST['job_id']) ? sanitizeInput($_POST['job_id']) : '';

// Validate required fields
if (empty($name) || empty($email) || empty($code) || empty($phone) || empty($job_id)) {
    securityLog("Missing required fields in career form submission", "WARNING");
    echo "1"; // Error response
    exit;
}

// Validate email format
if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
    securityLog("Invalid email format: $email", "WARNING");
    echo "1"; // Error response
    exit;
}

// File upload handling
$uploadDir = '../uploads/';
$resumePath = '';

// Create upload directory if it doesn't exist
if (!file_exists($uploadDir)) {
    mkdir($uploadDir, 0755, true);
}

// Proceed only if file is uploaded
if (isset($_FILES['fileUploadimage']) && $_FILES['fileUploadimage']['error'] === UPLOAD_ERR_OK) {
    $file = $_FILES['fileUploadimage'];
    
    // Check file size (5MB max)
    if ($file['size'] > 5 * 1024 * 1024) {
        securityLog("File too large: " . $file['size'] . " bytes", "WARNING");
        echo "1"; // Error response
        exit;
    }
    
    // Get file extension and validate
    $fileExtension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    $allowedExtensions = ['pdf', 'doc', 'docx'];
    
    if (!in_array($fileExtension, $allowedExtensions)) {
        securityLog("Invalid file type: " . $fileExtension, "WARNING");
        echo "1"; // Error response
        exit;
    }
    
    // Generate a unique filename
    $newFileName = 'resume_' . time() . '_' . bin2hex(random_bytes(8)) . '.' . $fileExtension;
    $uploadPath = $uploadDir . $newFileName;
    
    // Move the file to the uploads directory
    if (move_uploaded_file($file['tmp_name'], $uploadPath)) {
        $resumePath = $uploadPath;
        securityLog("File uploaded successfully: $resumePath", "INFO");
    } else {
        securityLog("Failed to move uploaded file", "ERROR");
        echo "1"; // Error response
        exit;
    }
} else {
    securityLog("No file uploaded or upload error: " . (isset($_FILES['fileUploadimage']) ? $_FILES['fileUploadimage']['error'] : 'No file'), "WARNING");
    echo "1"; // Error response
    exit;
}

// Prepare to save to database
try {
    // Connect to database
    $db = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
    
    if ($db->connect_error) {
        throw new Exception("Database connection failed: " . $db->connect_error);
    }
    
    // Start transaction
    $db->begin_transaction();
    
    // Prepare statement for inserting into database
    $stmt = $db->prepare("INSERT INTO job_applications (name, email, phone_code, phone, resume_path, job_id, application_date) VALUES (?, ?, ?, ?, ?, ?, NOW())");
    
    if (!$stmt) {
        throw new Exception("Prepare failed: " . $db->error);
    }
    
    $stmt->bind_param("ssssss", $name, $email, $code, $phone, $resumePath, $job_id);
    
    // Execute the statement
    if (!$stmt->execute()) {
        throw new Exception("Execute failed: " . $stmt->error);
    }
    
    // Get the inserted ID for reference
    $applicationId = $db->insert_id;
    
    // Send email notification
    require_once '../PHPMailer/PHPMailerAutoload.php';
    
    $mail = new PHPMailer;
    $mail->isSMTP();
    $mail->Host = SMTP_HOST;
    $mail->SMTPAuth = true;
    $mail->Username = SMTP_USER;
    $mail->Password = SMTP_PASS;
    $mail->SMTPSecure = 'tls';
    $mail->Port = SMTP_PORT;
    
    $mail->setFrom(SMTP_FROM_EMAIL, SMTP_FROM_NAME);
    $mail->addAddress(ADMIN_EMAIL);
    $mail->addReplyTo($email, $name);
    
    // Add attachment
    $mail->addAttachment($resumePath);
    
    $mail->isHTML(true);
    $mail->Subject = 'New Job Application: #' . $applicationId;
    
    // Prepare email body
    $mailBody = "
    <html>
    <head>
        <title>New Job Application</title>
    </head>
    <body>
        <h2>New Job Application Received</h2>
        <p>Application ID: #{$applicationId}</p>
        <p>Date: " . date('Y-m-d H:i:s') . "</p>
        <p>Job ID: {$job_id}</p>
        <hr>
        <h3>Applicant Details:</h3>
        <p>Name: {$name}</p>
        <p>Email: {$email}</p>
        <p>Phone: +{$code} {$phone}</p>
        <p>Resume: Attached</p>
    </body>
    </html>
    ";
    
    $mail->Body = $mailBody;
    $mail->AltBody = strip_tags($mailBody);
    
    if (!$mail->send()) {
        securityLog("Email could not be sent. Mailer Error: " . $mail->ErrorInfo, "ERROR");
        throw new Exception("Email could not be sent");
    }
    
    // Commit transaction
    $db->commit();
    
    // Log success
    securityLog("Job application submitted successfully. ID: " . $applicationId, "INFO");
    
    // Success response
    $response = 0;
    
} catch (Exception $e) {
    // Rollback transaction if there was an error
    if (isset($db) && $db->ping()) {
        $db->rollback();
    }
    
    // Log error
    securityLog("Error in job application submission: " . $e->getMessage(), "ERROR");
    
    // Error response
    $response = 1;
    
} finally {
    // Close database connection
    if (isset($stmt)) {
        $stmt->close();
    }
    if (isset($db) && $db->ping()) {
        $db->close();
    }
}

// Return response to client
echo $response;
?>