<?php
// Set secure access flag for includes
define('SECURE_ACCESS', true);

// Include necessary security and configuration files
$basePath = dirname(dirname(__FILE__));
require_once($basePath . '/includes/config.php');
require_once($basePath . '/includes/security.php');

// Enable detailed error reporting and logging for debugging
error_reporting(E_ALL);
ini_set('display_errors', 0); // Don't show to user, but log them

// Create a detailed debug log function
function debugLog($message, $data = null) {
    $logFile = __DIR__ . "/../debug/career_form.log";
    $logDir = dirname($logFile);
    
    // Create logs directory if it doesn't exist
    if (!file_exists($logDir)) {
        mkdir($logDir, 0755, true);
    }
    
    $timestamp = date('Y-m-d H:i:s');
    $logMessage = "[$timestamp] $message";
    
    if ($data !== null) {
        $logMessage .= " - Data: " . print_r($data, true);
    }
    
    $logMessage .= PHP_EOL;
    error_log($logMessage, 3, $logFile);
}

// Log the start of processing and the raw POST data
debugLog("Processing career form submission", $_POST);
debugLog("Processing career form FILES", $_FILES);

// Log function for security related events
function securityLog($message, $level = 'INFO') {
    $logFile = __DIR__ . "/../logs/security.log";
    $logDir = dirname($logFile);
    
    // Create logs directory if it doesn't exist
    if (!file_exists($logDir)) {
        mkdir($logDir, 0755, true);
    }
    
    $timestamp = date('Y-m-d H:i:s');
    $logMessage = "[$timestamp] [$level] $message" . PHP_EOL;
    error_log($logMessage, 3, $logFile);
}

// Validate CSRF token
function validateCSRFToken($token) {
    if (!isset($token)) {
        securityLog("CSRF token missing in form submission", "WARNING");
        return false;
    }
    
    $tokenValid = false;
    
    // If session based tokens
    if (isset($_SESSION['csrf_token'])) {
        $tokenValid = ($token === $_SESSION['csrf_token']);
    } 
    // Fallback to other validation methods if needed
    else if (function_exists('validateToken')) {
        $tokenValid = validateToken($token);
    }
    
    if (!$tokenValid) {
        securityLog("Invalid CSRF token: " . $token, "WARNING");
    }
    
    return $tokenValid;
}

// Sanitize input function
function sanitizeInput($input) {
    $input = trim($input);
    $input = stripslashes($input);
    $input = htmlspecialchars($input, ENT_QUOTES, 'UTF-8');
    return $input;
}

// Initialize response
$response = 0;

// Start session if not already started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Check if request method is POST
if ($_SERVER["REQUEST_METHOD"] != "POST") {
    securityLog("Invalid request method: " . $_SERVER["REQUEST_METHOD"], "WARNING");
    die(json_encode(["error" => "Invalid request method"]));
}

// Validate CSRF token
if (!isset($_POST['csrf_token']) || !validateCSRFToken($_POST['csrf_token'])) {
    // CSRF token validation failed
    securityLog("CSRF token validation failed", "WARNING");
    echo '1'; // Error response
    exit;
}

// Validate and sanitize input data
$name = isset($_POST['c_name']) ? sanitizeInput($_POST['c_name']) : '';
$email = isset($_POST['c_email']) ? filter_var(sanitizeInput($_POST['c_email']), FILTER_SANITIZE_EMAIL) : '';
$code = isset($_POST['c_code']) ? sanitizeInput($_POST['c_code']) : '';
$phone = isset($_POST['c_phone']) ? sanitizeInput($_POST['c_phone']) : '';
$position = isset($_POST['c_position']) ? sanitizeInput($_POST['c_position']) : '';
$country = isset($_POST['c_selected']) ? sanitizeInput($_POST['c_selected']) : '';

// Log input data for debugging
securityLog("Received form data - Name: $name, Email: $email, Code: $code, Phone: $phone, Position: $position, Country: $country", "INFO");

// Validate required fields
if (empty($name) || empty($email) || empty($code) || empty($phone) || empty($position)) {
    securityLog("Missing required fields in career form submission", "WARNING");
    echo "1"; // Error response
    exit;
}

// Validate email format
if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
    securityLog("Invalid email format: $email", "WARNING");
    echo "1"; // Error response
    exit;
}

// File upload handling
$uploadDir = '../uploads/';
$resumePath = '';

debugLog("Starting file upload handling");

// Create upload directory if it doesn't exist
if (!file_exists($uploadDir)) {
    mkdir($uploadDir, 0755, true);
    debugLog("Created uploads directory: " . $uploadDir);
}

// Add .htaccess to prevent direct access to uploads
$htaccess_file = $uploadDir . '.htaccess';
if (!file_exists($htaccess_file)) {
    file_put_contents($htaccess_file, "Order deny,allow\nDeny from all");
    debugLog("Created .htaccess file in uploads directory");
}

debugLog("Checking for file uploads", isset($_FILES['attachment']) ? "attachment exists" : "no attachment found");

// Check if files were uploaded
if (isset($_FILES['attachment']) && is_array($_FILES['attachment']['name'])) {
    debugLog("Attachment array structure", $_FILES['attachment']);
    
    // Define maxsize for files - 5MB
    $maxsize = MAX_UPLOAD_SIZE;
    $allowed_types = array('pdf', 'doc', 'docx');
    $allowed_mimes = array(
        'application/pdf', 
        'application/msword', 
        'application/vnd.openxmlformats-officedocument.wordprocessingml.document'
    );
    
    // Flag to track successful file upload
    $fileUploadSuccess = false;
    
    // Loop through each file in files[] array
    foreach ($_FILES['attachment']['tmp_name'] as $key => $value) {
        debugLog("Processing file #" . $key);
        
        // Check if file is actually uploaded
        if (is_uploaded_file($_FILES['attachment']['tmp_name'][$key])) {
            debugLog("File is uploaded properly");
            $file_tmpname = $_FILES['attachment']['tmp_name'][$key];
            $file_name = $_FILES['attachment']['name'][$key];
            $file_size = $_FILES['attachment']['size'][$key];
            $file_ext = strtolower(pathinfo($file_name, PATHINFO_EXTENSION));
            
            debugLog("File details", [
                'name' => $file_name,
                'size' => $file_size,
                'extension' => $file_ext,
                'tmp_name' => $file_tmpname
            ]);
            
            // Verify file size
            if ($file_size > $maxsize) {
                debugLog("File too large: " . $file_size . " bytes (max: " . $maxsize . ")");
                securityLog("File too large: " . $file_size . " bytes", "WARNING");
                continue; // Skip this file
            }
            
            // Check file type is allowed
            if (!in_array($file_ext, $allowed_types)) {
                debugLog("Invalid file type: " . $file_ext);
                securityLog("Invalid file type: " . $file_ext, "WARNING");
                continue; // Skip this file
            }
            
            // Additional MIME type check for security
            $finfo = new finfo(FILEINFO_MIME_TYPE);
            $file_mime = $finfo->file($file_tmpname);
            debugLog("File MIME type: " . $file_mime);
            
            if (!in_array($file_mime, $allowed_mimes)) {
                debugLog("Invalid file MIME type: " . $file_mime);
                securityLog("Invalid file MIME type: " . $file_mime, "WARNING");
                continue; // Skip this file
            }
            
            // Generate a secure random filename while preserving extension
            $secure_filename = 'resume_' . time() . '_' . bin2hex(random_bytes(8)) . '.' . $file_ext;
            $uploadPath = $uploadDir . $secure_filename;
            debugLog("Generated secure filename: " . $secure_filename);
            
            debugLog("Attempting to move uploaded file to: " . $uploadPath);
            if (move_uploaded_file($file_tmpname, $uploadPath)) {
                // Set appropriate permissions on the file
                chmod($uploadPath, 0644);
                
                $resumePath = $uploadPath;
                $fileUploadSuccess = true;
                debugLog("File uploaded successfully: " . $resumePath);
                securityLog("File uploaded successfully: " . $resumePath, "INFO");
            } else {
                debugLog("Failed to move uploaded file. PHP error: " . error_get_last()['message']);
                securityLog("Failed to move uploaded file", "ERROR");
            }
        } else {
            debugLog("File was not uploaded properly via HTTP POST");
        }
    }
    
    if (!$fileUploadSuccess) {
        debugLog("No valid files were uploaded");
        securityLog("No valid files were uploaded", "WARNING");
        echo "1"; // Error response
        exit;
    }
} else {
    debugLog("No files selected for upload. FILES array:", $_FILES);
    securityLog("No files selected for upload", "WARNING");
    echo "1"; // Error response
    exit;
}

// Prepare to send email and save to database
try {
    // Send email notification
    require_once '../PHPMailer/PHPMailerAutoload.php';
    
    $mail = new PHPMailer;
    $mail->isSMTP();
    $mail->Host = SMTP_HOST;
    $mail->SMTPAuth = true;
    $mail->Username = SMTP_USER;
    $mail->Password = SMTP_PASS;
    $mail->SMTPSecure = SMTP_SECURE;
    $mail->Port = SMTP_PORT;
    
    $mail->setFrom(SMTP_FROM_EMAIL, SMTP_FROM_NAME);
    $mail->addAddress(ADMIN_EMAIL);
    $mail->addCC('mankesh.dadhwal@caeliusconsulting.com');
    $mail->addReplyTo($email, $name);
    
    // Add attachment
    $mail->addAttachment($resumePath);
    
    $mail->isHTML(true);
    $mail->Subject = "Resume " . $name . " " . $position;
    
    // Prepare email body
    $mailBody = "
    <html>
    <head>
        <title>New Career Application</title>
    </head>
    <body>
        <h2>New Career Application Received</h2>
        <p>Date: " . date('Y-m-d H:i:s') . "</p>
        <hr>
        <h3>Applicant Details:</h3>
        <p><b>Name:</b> {$name}</p>
        <p><b>CountryCode:</b> {$code}</p>
        <p><b>PhoneNo:</b> {$phone}</p>
        <p><b>Email:</b> {$email}</p>
        <p><b>Position:</b> {$position}</p>
        <p><b>Country:</b> {$country}</p>
    </body>
    </html>
    ";
    
    $mail->Body = $mailBody;
    $mail->AltBody = strip_tags($mailBody);
    
    if (!$mail->send()) {
        securityLog("Email could not be sent. Mailer Error: " . $mail->ErrorInfo, "ERROR");
        throw new Exception("Email could not be sent");
    }
    
    // Connect to database and save the application
    require_once 'db_config.php';
    $conn = getDbConnection();
    
    if (!$conn) {
        debugLog("Database connection failed");
        throw new Exception("Failed to connect to database");
    }
    
    debugLog("Database connection successful");
    
    // Start transaction
    $conn->begin_transaction();
    debugLog("Transaction started");
    
    // Check if the table exists
    $tableCheck = $conn->query("SHOW TABLES LIKE 'per_resume'");
    if ($tableCheck->num_rows == 0) {
        debugLog("Table 'per_resume' does not exist");
        throw new Exception("Table 'per_resume' does not exist");
    }
    
    debugLog("Table 'per_resume' exists");
    
    // Use prepared statements
    $stmt = $conn->prepare("INSERT INTO `per_resume`(`Name`, `Email`, `Country_Code`, `Phone_No`, `Resume`, `Country_Selected`, `Position_Name`) 
           VALUES (?, ?, ?, ?, ?, ?, ?)");
    
    if (!$stmt) {
        debugLog("Prepare statement failed: " . $conn->error);
        throw new Exception("Prepare statement failed: " . $conn->error);
    }
    
    debugLog("Prepare statement successful");
    debugLog("Binding parameters", [
        'Name' => $name,
        'Email' => $email, 
        'Country_Code' => $code, 
        'Phone_No' => $phone, 
        'Resume' => $resumePath, 
        'Country_Selected' => $country, 
        'Position_Name' => $position
    ]);
    
    $stmt->bind_param("sssssss", 
        $name,
        $email,
        $code,
        $phone,
        $resumePath,
        $country,
        $position
    );
    
    if ($stmt->execute()) {
        // Commit the transaction
        $conn->commit();
        debugLog("Database record created successfully with ID: " . $conn->insert_id);
        securityLog("Database record created successfully", "INFO");
        $response = 2; // Success response
    } else {
        // Rollback in case of error
        $conn->rollback();
        debugLog("Database execution error: " . $stmt->error);
        securityLog("Database error: " . $stmt->error, "ERROR");
        $response = 1; // Error response
    }
    
} catch (Exception $e) {
    // Rollback transaction if there was an error
    if (isset($conn) && $conn->ping()) {
        $conn->rollback();
    }
    
    // Log error
    debugLog("Exception caught: " . $e->getMessage());
    securityLog("Error in career application submission: " . $e->getMessage(), "ERROR");
    
    // Error response
    $response = 1;
    
} finally {
    // Close database connection
    if (isset($stmt)) {
        $stmt->close();
    }
    if (isset($conn) && $conn->ping()) {
        $conn->close();
    }
}

// Return response to client
debugLog("Sending response to client: " . $response);
echo $response;
?> 