<?php
// Set secure access flag for includes
define('SECURE_ACCESS', true);

// Include necessary files
$basePath = dirname(dirname(__FILE__));
require_once($basePath . '/includes/config.php');
require_once($basePath . '/includes/security.php');
require_once($basePath . '/mail/db_config.php');

// If debug mode is enabled
$debug = DEBUG_MODE;

/**
 * Log security events
 * 
 * @param string $message The message to log
 * @param string $level The level (INFO, WARNING, ERROR)
 * @return void
 */
function securityLog($message, $level = 'INFO') {
    $timestamp = date('Y-m-d H:i:s');
    $ipAddress = isset($_SERVER['REMOTE_ADDR']) ? $_SERVER['REMOTE_ADDR'] : 'Unknown';
    $userAgent = isset($_SERVER['HTTP_USER_AGENT']) ? $_SERVER['HTTP_USER_AGENT'] : 'Unknown';
    
    $logEntry = "[{$timestamp}] [{$level}] [{$ipAddress}] [{$userAgent}] {$message}" . PHP_EOL;
    
    // Log to file
    $logFile = dirname(dirname(__FILE__)) . '/logs/security.log';
    
    // Create directory if it doesn't exist
    if (!is_dir(dirname($logFile))) {
        mkdir(dirname($logFile), 0755, true);
    }
    
    file_put_contents($logFile, $logEntry, FILE_APPEND);
}

/**
 * Validate CSRF token
 * 
 * @param string $token The token to validate
 * @return bool Whether the token is valid
 */
function validateCSRFToken($token) {
    if (!isset($_SESSION['csrf_token'])) {
        securityLog("CSRF token not found in session", "WARNING");
        return false;
    }
    
    if (empty($token)) {
        securityLog("Empty CSRF token submitted", "WARNING");
        return false;
    }
    
    if (!hash_equals($_SESSION['csrf_token'], $token)) {
        securityLog("Invalid CSRF token: " . $token, "WARNING");
        return false;
    }
    
    return true;
}

/**
 * Sanitize input
 * 
 * @param string $input The input to sanitize
 * @return string Sanitized input
 */
function sanitizeInput($input) {
    $input = trim($input);
    $input = stripslashes($input);
    $input = htmlspecialchars($input, ENT_QUOTES, 'UTF-8');
    return $input;
}

// Initialize response
$response = array(
    'status' => 'error',
    'message' => 'An error occurred while processing your request.'
);

// Start session if not already started
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// Only process POST requests
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Validate CSRF token
    if (!isset($_POST['csrf_token']) || !validateCSRFToken($_POST['csrf_token'])) {
        $response['message'] = 'Invalid security token. Please refresh the page and try again.';
        securityLog("Contact form submission rejected due to invalid CSRF token", "WARNING");
        echo json_encode($response);
        exit;
    }
    
    // Validate required fields
    $requiredFields = array('name', 'email', 'phone', 'subject', 'message');
    $missingFields = array();
    
    foreach ($requiredFields as $field) {
        if (!isset($_POST[$field]) || empty($_POST[$field])) {
            $missingFields[] = $field;
        }
    }
    
    if (!empty($missingFields)) {
        $response['message'] = 'The following fields are required: ' . implode(', ', $missingFields);
        echo json_encode($response);
        exit;
    }
    
    // Sanitize input
    $name = sanitizeInput($_POST['name']);
    $email = sanitizeInput($_POST['email']);
    $phone = sanitizeInput($_POST['phone']);
    $subject = sanitizeInput($_POST['subject']);
    $message = sanitizeInput($_POST['message']);
    
    // Validate email
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $response['message'] = 'Please enter a valid email address.';
        echo json_encode($response);
        exit;
    }
    
    // Use PHPMailer to send email
    try {
        // Path to PHPMailer
        require_once '../PHPMailer/PHPMailerAutoload.php';
        
        // Create new PHPMailer instance
        $mail = new PHPMailer;
        
        // Server settings
        $mail->isSMTP();
        $mail->Host = SMTP_HOST;
        $mail->SMTPAuth = true;
        $mail->Username = SMTP_USER;
        $mail->Password = SMTP_PASS;
        $mail->SMTPSecure = SMTP_SECURE;
        $mail->Port = SMTP_PORT;
        
        // Recipients
        $mail->setFrom(SMTP_FROM_EMAIL, SMTP_FROM_NAME);
        $mail->addAddress(CONTACT_EMAIL, CONTACT_NAME);
        $mail->addReplyTo($email, $name);
        
        // Content
        $mail->isHTML(true);
        $mail->Subject = 'Contact Form: ' . $subject;
        
        // Build email body
        $mail->Body = "
            <html>
            <head>
                <title>Contact Form Submission</title>
            </head>
            <body>
                <h2>Contact Form Submission</h2>
                <p><strong>Name:</strong> " . $name . "</p>
                <p><strong>Email:</strong> " . $email . "</p>
                <p><strong>Phone:</strong> " . $phone . "</p>
                <p><strong>Subject:</strong> " . $subject . "</p>
                <p><strong>Message:</strong></p>
                <p>" . nl2br($message) . "</p>
                <p><small>This email was sent from the contact form on the Caelius Consulting website.</small></p>
            </body>
            </html>
        ";
        
        $mail->AltBody = "
            Contact Form Submission
            
            Name: {$name}
            Email: {$email}
            Phone: {$phone}
            Subject: {$subject}
            
            Message:
            {$message}
            
            This email was sent from the contact form on the Caelius Consulting website.
        ";
        
        // Save to database
        $conn = getDbConnection();
        if ($conn) {
            // Prepare statement
            $stmt = $conn->prepare("INSERT INTO contact_submissions (name, email, phone, subject, message, ip_address, submission_date) VALUES (?, ?, ?, ?, ?, ?, NOW())");
            
            if ($stmt) {
                $ipAddress = isset($_SERVER['REMOTE_ADDR']) ? $_SERVER['REMOTE_ADDR'] : 'Unknown';
                $stmt->bind_param("ssssss", $name, $email, $phone, $subject, $message, $ipAddress);
                $stmt->execute();
                $stmt->close();
            } else {
                // Log database error
                securityLog("Failed to prepare statement for contact submission: " . $conn->error, "ERROR");
            }
            
            // Close connection
            closeDbConnection($conn);
        }
        
        // Send the email
        $mail->send();
        
        // Log successful submission
        securityLog("Contact form submitted successfully by {$name} ({$email})", "INFO");
        
        // Update response
        $response['status'] = 'success';
        $response['message'] = 'Your message has been sent. Thank you!';
    } catch (Exception $e) {
        // Log error
        securityLog("Failed to send contact form: " . $e->getMessage(), "ERROR");
        
        if ($debug) {
            $response['message'] = 'Message could not be sent. Mailer Error: ' . $e->getMessage();
        } else {
            $response['message'] = 'Message could not be sent. Please try again later.';
        }
    }
}

// Return response as JSON
echo json_encode($response);
?>