<?php
/**
 * Security Functions
 * This file contains essential security functions for the website
 */

// Start session properly if it's not already active
if (!headers_sent() && session_status() == PHP_SESSION_NONE) {
    // Start session if not already started
    session_start();
} else {
    // If headers are already sent, just make sure the session is started
    if (session_status() == PHP_SESSION_NONE) {
        @session_start();
    }
}

// Include the XSS protection functions
if (file_exists(__DIR__ . '/xss-protection.php')) {
    require_once __DIR__ . '/xss-protection.php';
} 

// Include the SRI implementation
if (file_exists(__DIR__ . '/security/sri-implementation.php')) {
    require_once __DIR__ . '/security/sri-implementation.php';
}

/**
 * Generate a CSRF token for a specific form
 * 
 * @param string $form_name Name of the form to generate token for
 * @return string Generated token
 */
function generateCSRFToken($form_name = 'default') {
    // Initialize the token array if it doesn't exist
    if (!isset($_SESSION['csrf_tokens'])) {
        $_SESSION['csrf_tokens'] = [];
    }
    
    // Check if token already exists for this form
    if (!isset($_SESSION['csrf_tokens'][$form_name])) {
        // Generate a new token
        $token = bin2hex(random_bytes(32));
        $_SESSION['csrf_tokens'][$form_name] = [
            'token' => $token,
            'timestamp' => time()
        ];
    }
    
    return $_SESSION['csrf_tokens'][$form_name]['token'];
}

/**
 * Validate a CSRF token
 * 
 * @param string $token Token to validate
 * @param string $form_name Name of the form to validate token for
 * @param int $expiry_time Token expiry time in seconds
 * @return bool True if token is valid, false otherwise
 */
function validateCSRFToken($token, $form_name = 'default', $expiry_time = 3600) {
    // Check if token exists
    if (!isset($_SESSION['csrf_tokens']) || !isset($_SESSION['csrf_tokens'][$form_name])) {
        return false;
    }
    
    $stored_token = $_SESSION['csrf_tokens'][$form_name];
    
    // Check if token has expired
    if (time() - $stored_token['timestamp'] > $expiry_time) {
        // Token has expired, remove it
        unset($_SESSION['csrf_tokens'][$form_name]);
        return false;
    }
    
    // Compare tokens
    if ($token === $stored_token['token']) {
        // Token is valid, remove it to prevent reuse (optional)
        // unset($_SESSION['csrf_tokens'][$form_name]);
        return true;
    }
    
    return false;
}

/**
 * Generate HTML for a CSRF token input field
 * 
 * @param string $form_name Name of the form
 * @return string HTML input field
 */
function csrfTokenField($form_name = 'default') {
    $token = generateCSRFToken($form_name);
    return '<input type="hidden" name="csrf_token" value="' . escape_output($token) . '">';
}

/**
 * Verify CSRF token from POST request
 * 
 * @param string $form_name Name of the form
 * @return bool True if token is valid, false otherwise
 */
function verifyCSRFToken($form_name = 'default') {
    if (!isset($_POST['csrf_token'])) {
        return false;
    }
    
    return validateCSRFToken($_POST['csrf_token'], $form_name);
}

// XSS protection fallbacks if the xss-protection.php file is missing
if (!function_exists('sanitizeOutput')) {
    function sanitizeOutput($data) {
        if (is_array($data)) {
            foreach ($data as $key => $value) {
                $data[$key] = sanitizeOutput($value);
            }
            return $data;
        }
        return htmlspecialchars($data, ENT_QUOTES, 'UTF-8');
    }
}

if (!function_exists('sanitizeInput')) {
    function sanitizeInput($data) {
        if (is_array($data)) {
            foreach ($data as $key => $value) {
                $data[$key] = sanitizeInput($value);
            }
            return $data;
        }
        $data = trim($data);
        $data = stripslashes($data);
        return $data;
    }
}

if (!function_exists('escape_output')) {
    function escape_output($data) {
        return sanitizeOutput($data);
    }
}

if (!function_exists('safe_echo')) {
    function safe_echo($data) {
        echo escape_output($data);
    }
}

if (!function_exists('safe_output')) {
    function safe_output($data) {
        return escape_output($data);
    }
}

if (!function_exists('sanitize_url')) {
    function sanitize_url($url) {
        // Remove invalid characters
        $url = filter_var($url, FILTER_SANITIZE_URL);
        
        // Only allow http, https, and relative URLs
        if ($url !== '' && $url[0] !== '/' && 
            strpos($url, 'http://') !== 0 && 
            strpos($url, 'https://') !== 0) {
            return '';
        }
        
        return $url;
    }
}

if (!function_exists('safe_js')) {
    function safe_js($src) {
        $src = sanitize_url($src);
        if (empty($src)) {
            return;
        }
        
        echo '<script src="' . escape_output($src) . '"></script>' . "\n";
    }
}

if (!function_exists('safe_css')) {
    function safe_css($href) {
        $href = sanitize_url($href);
        if (empty($href)) {
            return;
        }
        
        echo '<link rel="stylesheet" href="' . escape_output($href) . '">' . "\n";
    }
}

/**
 * Verify an uploaded file for security
 * 
 * @param array $file The $_FILES element to verify
 * @param array $allowed_types Array of allowed MIME types
 * @param array $allowed_extensions Array of allowed file extensions
 * @param int $max_size Maximum file size in bytes
 * @return array|bool Array with file info if valid, false if invalid
 */
function verifyUploadedFile($file, $allowed_types, $allowed_extensions, $max_size = 2097152) {
    // Basic upload validation
    if (!isset($file) || $file['error'] !== UPLOAD_ERR_OK) {
        return false;
    }
    
    // Check file size
    if ($file['size'] > $max_size) {
        return false;
    }
    
    // Get file info
    $file_info = pathinfo($file['name']);
    $extension = strtolower($file_info['extension']);
    
    // Check file extension
    if (!in_array($extension, $allowed_extensions)) {
        return false;
    }
    
    // Check MIME type
    $finfo = new finfo(FILEINFO_MIME_TYPE);
    $mime_type = $finfo->file($file['tmp_name']);
    
    if (!in_array($mime_type, $allowed_types)) {
        return false;
    }
    
    // Additional security checks
    // Check for PHP code in the file
    $file_content = file_get_contents($file['tmp_name']);
    if (preg_match('/<\?php/i', $file_content)) {
        return false;
    }
    
    // Return file information if it passes all checks
    return [
        'name' => $file['name'],
        'tmp_name' => $file['tmp_name'],
        'extension' => $extension,
        'mime_type' => $mime_type,
        'size' => $file['size']
    ];
}

/**
 * Generate a secure filename for uploaded files
 * 
 * @param string $original_filename Original filename
 * @param string $extension File extension
 * @return string Secure filename
 */
function generateSecureFilename($original_filename, $extension) {
    // Generate a random string
    $random = bin2hex(random_bytes(16));
    
    // Sanitize the original filename (remove spaces, special chars)
    $sanitized = preg_replace('/[^a-zA-Z0-9]/', '_', pathinfo($original_filename, PATHINFO_FILENAME));
    $sanitized = substr($sanitized, 0, 20); // Limit length
    
    // Combine and return
    return $sanitized . '_' . $random . '.' . $extension;
}

// SRI Functions (simplified versions if sri-implementation.php is missing)
if (!function_exists('sriScriptTag')) {
    function sriScriptTag($src, $integrity = null, $attrs = []) {
        $src = sanitize_url($src);
        if (empty($src)) {
            return;
        }
        
        echo '<script src="' . escape_output($src) . '"';
        if ($integrity) {
            echo ' integrity="' . escape_output($integrity) . '" crossorigin="anonymous"';
        }
        echo '></script>' . "\n";
    }
}

if (!function_exists('sriLinkTag')) {
    function sriLinkTag($href, $integrity = null, $attrs = []) {
        $href = sanitize_url($href);
        if (empty($href)) {
            return;
        }
        
        echo '<link rel="stylesheet" href="' . escape_output($href) . '"';
        if ($integrity) {
            echo ' integrity="' . escape_output($integrity) . '" crossorigin="anonymous"';
        }
        echo '>' . "\n";
    }
}