<?php
/**
 * Subresource Integrity (SRI) Implementation
 * Functions to add integrity hashes to external CSS and JavaScript resources
 */

// SRI hashes for common CDN resources
$sriHashes = [
    // jQuery
    'https://code.jquery.com/jquery-3.6.0.min.js' => 'sha384-vtXRMe3mGCbOeY7l30aIg8H9p3GdeSe4IFlP6G8JMa7o7lXvnz3GFKzPxzJdPfGK',
    'https://code.jquery.com/jquery-3.5.1.min.js' => 'sha384-ZvpUoO/+PpLXR1lu4jmpXWu80pZlYUAfxl5NsBMWOEPSjUn/6Z/hRTt8+pR6L4N2',
    'https://ajax.googleapis.com/ajax/libs/jquery/3.1.1/jquery.min.js' => 'sha384-3ceskX3iaEnIogmQchP8opvBy3Mi7Ce34nWjpBIwVTHfGYWQS9jwHDVRnpKKHJg7',
    
    // Bootstrap
    'https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.min.js' => 'sha384-QJHtvGhmr9XOIpI6YVutG+2QOK9T+ZnN4kzFN1RtK3zEFEIsxhlmWl5/YESvpZ13',
    'https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css' => 'sha384-1BmE4kWBq78iYhFldvKuhfTAU6auU8tT94WrHftjDbrCEXSU1oBoqyl2QvZ6jIW3',
    'https://cdn.jsdelivr.net/npm/bootstrap@5.0.2/dist/js/bootstrap.min.js' => 'sha384-cVKIPhGWiC2Al4u+LWgxfKTRIcfu0JTxR+EQDz/bgldoEyl4H0zUF0QKbrJ0EcQF',
    'https://cdn.jsdelivr.net/npm/bootstrap@5.0.2/dist/css/bootstrap.min.css' => 'sha384-EVSTQN3/azprG1Anm3QDgpJLIm9Nao0Yz1ztcQTwFspd3yD65VohhpuuCOmLASjC',
   
    // AOS
    'https://unpkg.com/aos@2.3.1/dist/aos.js' => 'sha384-wziAfuz/UgZ2ALLXPa5ilEkl4UW9xDiriEMWkFbOlaZpxGZna0TYjuRLksjMpwxH',
    'https://unpkg.com/aos@2.3.1/dist/aos.css' => 'sha384-r/k8v3wEYw8jRcH3f0nQ+m/1zZ+/8qTJzw+Gx2o7MpMbY/YHVHWoGZFMPp2xfTF9',
    
    // Popper
    'https://cdn.jsdelivr.net/npm/popper.js@1.16.0/dist/umd/popper.min.js' => 'sha384-Q6E9RHvbIyZFJoft+2mJbHaEWldlvI9IOYy5n3zV9zzTtmI3UksdQRVvoxMfooAo',
    
    // Font Awesome
    'https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css' => 'sha384-DyZ88mC6Up2uqS4h/KRgHuoeGwBcD4Ng9SiP4dIRy0EXTlnuz47vAwmeGwVChigm',
    
    // Bootstrap Datepicker
    'https://cdnjs.cloudflare.com/ajax/libs/bootstrap-datepicker/1.9.0/css/bootstrap-datepicker.min.css' => 'sha384-E486E0B1DAAB8C5EAB017D66C41664291C943A012BD3FC421A4B5E2486912A9BD6D00834B5FEA5AEEE282F666170CBE8',
    'https://cdnjs.cloudflare.com/ajax/libs/bootstrap-datepicker/1.9.0/js/bootstrap-datepicker.min.js' => 'sha384-duuqMKp8/CZ6ncoiIGbR4GkR9uqSCYxQqnUMnG/dP2/iw4SRqXbcQxvewz9MyMLG',
    
    // Bootstrap Datetimepicker
    'https://cdnjs.cloudflare.com/ajax/libs/bootstrap-datetimepicker/4.17.37/css/bootstrap-datetimepicker.min.css' => 'sha384-FD79A145C07A7B9D41DCD3B54B95555A734A08333BE0D957D1C9B1654C0224F8363FFD040939012A78E7CCD9C5EEB5F9',
    'https://cdnjs.cloudflare.com/ajax/libs/bootstrap-datetimepicker/4.17.37/js/bootstrap-datetimepicker.min.js' => 'sha384-0CA0DA7ADE9E7AEA48B30475171DB3C7D1646E9FE8ED819FC901790DB0D95E9F694F351FB085DCCDC432A0C276C15A6E',
    
    // Google Tag Manager
    'https://www.googletagmanager.com/gtag/js?id=UA-129311972-1' => null, // Google resources are often dynamic, so no hash
];

/**
 * Create a script tag with integrity attribute
 * 
 * @param string $src The URL of the script
 * @param string|null $integrity The integrity hash (optional, will use from known list if null)
 * @param array $attributes Additional attributes for the script tag
 * @return string The generated script tag
 */
function sriScriptTag($src, $integrity = null, $attributes = []) {
    global $sriHashes;
    
    if ($integrity === null && isset($sriHashes[$src])) {
        $integrity = $sriHashes[$src];
    }
    
    $attrStr = '';
    foreach ($attributes as $name => $value) {
        if ($value === true) {
            $attrStr .= " $name";
        } else {
            $attrStr .= " $name=\"" . htmlspecialchars($value, ENT_QUOTES, 'UTF-8') . "\"";
        }
    }
    
    $tag = "<script src=\"" . htmlspecialchars($src, ENT_QUOTES, 'UTF-8') . "\"";
    
    if ($integrity) {
        $tag .= " integrity=\"" . htmlspecialchars($integrity, ENT_QUOTES, 'UTF-8') . "\" crossorigin=\"anonymous\"";
    }
    
    $tag .= $attrStr . "></script>";
    
    return $tag;
}

/**
 * Create a link tag with integrity attribute
 * 
 * @param string $href The URL of the stylesheet
 * @param string|null $integrity The integrity hash (optional, will use from known list if null)
 * @param array $attributes Additional attributes for the link tag
 * @return string The generated link tag
 */
function sriLinkTag($href, $integrity = null, $attributes = []) {
    global $sriHashes;
    
    if ($integrity === null && isset($sriHashes[$href])) {
        $integrity = $sriHashes[$href];
    }
    
    $attrStr = '';
    foreach ($attributes as $name => $value) {
        if ($value === true) {
            $attrStr .= " $name";
        } else {
            $attrStr .= " $name=\"" . htmlspecialchars($value, ENT_QUOTES, 'UTF-8') . "\"";
        }
    }
    
    $tag = "<link rel=\"stylesheet\" href=\"" . htmlspecialchars($href, ENT_QUOTES, 'UTF-8') . "\"";
    
    if ($integrity) {
        $tag .= " integrity=\"" . htmlspecialchars($integrity, ENT_QUOTES, 'UTF-8') . "\" crossorigin=\"anonymous\"";
    }
    
    $tag .= $attrStr . ">";
    
    return $tag;
}

/**
 * Utility function to generate SRI hash for a file
 * Not typically used in production, but useful for development
 * 
 * @param string $url The URL of the file to generate a hash for
 * @param string $algo The algorithm to use (sha384 recommended)
 * @return string|false The generated hash or false on failure
 */
function generateSriHash($url, $algo = 'sha384') {
    $content = @file_get_contents($url);
    if ($content === false) {
        return false;
    }
    
    $hash = hash($algo, $content, true);
    return "$algo-" . base64_encode($hash);
} 