<?php
/**
 * XSS Protection Functions
 * This file contains functions to prevent cross-site scripting (XSS) attacks
 */

/**
 * Sanitize output to prevent XSS
 * 
 * @param string $data The data to sanitize
 * @return string The sanitized data
 */
function escape_output($data) {
    if (is_array($data)) {
        foreach ($data as $key => $value) {
            $data[$key] = escape_output($value);
        }
        return $data;
    }
    
    return htmlspecialchars($data, ENT_QUOTES, 'UTF-8');
}

/**
 * Safe way to output data in HTML
 * 
 * @param string $data The data to output
 * @return void
 */
function safe_echo($data) {
    echo escape_output($data);
}

/**
 * Safe way to include HTML content that is trusted
 * Use this function only for content that needs to preserve HTML
 * 
 * @param string $html The HTML to output
 * @return void
 */
function safe_html($html) {
    // This function should only be used for known safe HTML content
    // Typically content created by administrators or from trusted sources
    // Never use this function for user-submitted content
    echo $html;
}

/**
 * Sanitize URL to prevent XSS through URL parameters
 * 
 * @param string $url The URL to sanitize
 * @return string The sanitized URL
 */
function sanitize_url($url) {
    // Remove invalid characters
    $url = filter_var($url, FILTER_SANITIZE_URL);
    
    // Only allow http, https, and relative URLs
    if ($url !== '' && $url[0] !== '/' && 
        strpos($url, 'http://') !== 0 && 
        strpos($url, 'https://') !== 0) {
        return '';
    }
    
    return $url;
}

/**
 * Sanitize user input for safe database insertion
 * 
 * @param string $data The data to sanitize
 * @return string The sanitized data
 */
function sanitize_input($data) {
    $data = trim($data);
    $data = stripslashes($data);
    return $data;
}

/**
 * Safely include an external JavaScript resource with SRI
 * 
 * @param string $src The source URL
 * @param string $integrity The SRI integrity hash
 * @return void
 */
function safe_js($src, $integrity = '') {
    $src = sanitize_url($src);
    if (empty($src)) {
        return;
    }
    
    if (!empty($integrity)) {
        echo '<script src="' . escape_output($src) . '" integrity="' . 
            escape_output($integrity) . '" crossorigin="anonymous"></script>' . "\n";
    } else {
        echo '<script src="' . escape_output($src) . '"></script>' . "\n";
    }
}

/**
 * Safely include an external CSS resource with SRI
 * 
 * @param string $href The source URL
 * @param string $integrity The SRI integrity hash
 * @return void
 */
function safe_css($href, $integrity = '') {
    $href = sanitize_url($href);
    if (empty($href)) {
        return;
    }
    
    if (!empty($integrity)) {
        echo '<link rel="stylesheet" href="' . escape_output($href) . '" integrity="' . 
            escape_output($integrity) . '" crossorigin="anonymous">' . "\n";
    } else {
        echo '<link rel="stylesheet" href="' . escape_output($href) . '">' . "\n";
    }
}